<?php

namespace App\Http\Controllers\API\Slider;

use App\Filters\Global\OrderByFilter;
use App\Http\Controllers\Controller;
use App\Http\Requests\API\Slider\SliderRequest;
use App\Http\Requests\Global\DeleteAllRequest;
use App\Http\Requests\Global\PageRequest;
use App\Http\Resources\API\Slider\SliderResource;
use App\Models\Slider;
use App\Services\Global\UploadService;
use Illuminate\Http\JsonResponse;
use Illuminate\Pipeline\Pipeline;

class SliderController extends Controller
{
    /**
     * @param PageRequest $request
     * @return JsonResponse
     */
    public function index(PageRequest $request): JsonResponse
    {
        $query = app(Pipeline::class)
            ->send(Slider::query())
            ->through([OrderByFilter::class])
            ->thenReturn();

        return successResponse(fetchData($query, $request->pageSize, SliderResource::class));
    }

    /**
     * @param SliderRequest $request
     * @return JsonResponse
     */
    public function store(SliderRequest $request): JsonResponse
    {
        $slider = Slider::create($request->validated());

        return successResponse(new SliderResource($slider->refresh()), __('api.created_success'));
    }

    /**
     * @param Slider $slider
     * @return JsonResponse
     */
    public function show(Slider $slider): JsonResponse
    {
        return successResponse(new SliderResource($slider));
    }

    /**
     * @param SliderRequest $request
     * @param Slider $slider
     * @return JsonResponse
     */
    public function update(SliderRequest $request, Slider $slider): JsonResponse
    {
        $slider->update($request->validated());

        return successResponse(new SliderResource($slider->refresh()), __('api.updated_success'));
    }

    /**
     * @param Slider $slider
     * @return JsonResponse
     */
    public function destroy(Slider $slider): JsonResponse
    {
        UploadService::delete($slider->image);
        $slider->delete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function destroyAll(DeleteAllRequest $request): JsonResponse
    {
        Slider::whereIn('id', $request->ids)->delete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function restoreAll(DeleteAllRequest $request): JsonResponse
    {
        Slider::whereIn('id', $request->ids)->onlyTrashed()->restore();

        return successResponse(msg: __('api.restored_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function forceDeleteAll(DeleteAllRequest $request): JsonResponse
    {
        Slider::whereIn('id', $request->ids)->onlyTrashed()->forceDelete();

        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param int $id
     * @return JsonResponse
     */
    public function forceDelete(int $id): JsonResponse
    {
        Slider::onlyTrashed()->findOrFail($id)->forceDelete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param int $id
     * @return JsonResponse
     */
    public function restore(int $id): JsonResponse
    {
        Slider::onlyTrashed()->findOrFail($id)?->restore();

        return successResponse(msg: __('api.restored_success'));
    }

}
