<?php

namespace App\Http\Controllers\API;

use App\Models\Product;
use Illuminate\Support\Arr;
use Illuminate\Http\JsonResponse;
use Illuminate\Pipeline\Pipeline;
use App\Filters\Global\NameFilter;
use App\Http\Controllers\Controller;
use App\Filters\Global\OrderByFilter;
use App\Http\Requests\Global\PageRequest;
use App\Http\Requests\API\ProductRequest;
use App\Http\Resources\API\ProductResource;
use App\Http\Requests\Global\DeleteAllRequest;

class ProductController extends Controller
{
    /**
     * @param PageRequest $request
     * @return JsonResponse
     */
    public function index(PageRequest $request): JsonResponse
    {
        $query = app(Pipeline::class)
            ->send(Product::with('files'))
            ->through([NameFilter::class, OrderByFilter::class])
            ->thenReturn();

        return successResponse(fetchData($query, $request->pageSize, ProductResource::class));
    }

    /**
     * @param ProductRequest $request
     * @return JsonResponse
     */
    public function store(ProductRequest $request): JsonResponse
    {
        $product = Product::create($request->validated());
        if ($request->files) {
            $product->syncFiles($request->file('files', []));
        }
        return successResponse(new ProductResource($product->refresh()->load(['category'])), __('api.created_success'));
    }

    /**
     * @param Product $product
     * @return JsonResponse
     */
    public function show(Product $product): JsonResponse
    {
        return successResponse(new ProductResource($product->load(['category'])));
    }

    /**
     * @param ProductRequest $request
     * @param Product $product
     * @return JsonResponse
     */
    public function update(ProductRequest $request, Product $product): JsonResponse
    {
        $product->update($request->validated());

        $product->syncFiles($request->file('files', []), true);

        return successResponse(new ProductResource($product->refresh()->load(['category'])), __('api.updated_success'));
    }

    /**
     * @param Product $product
     * @return JsonResponse
     */
    public function destroy(Product $product): JsonResponse
    {
        $product->files()->delete();
        $product->delete();
        return successResponse(msg: __('api.deleted_success'));
    }

    /**
     * @param DeleteAllRequest $request
     * @return JsonResponse
     */
    public function destroyAll(DeleteAllRequest $request): JsonResponse
    {
        Product::whereIn('id', $request->ids)->delete();
        return successResponse(msg: __('api.deleted_success'));
    }
}
